<?php
// app/Controllers/TaskController.php

namespace App\Controllers;

use App\Core\Request;
use App\Core\Response;
use App\Models\Task;
use App\Models\Log;
use App\Middleware\AuthMiddleware;
use App\Middleware\AdminMiddleware;

class TaskController {
    private $taskModel;
    private $logModel;

    public function __construct() {
        $this->taskModel = new Task();
        $this->logModel = new Log();
    }

    public function index() {
        AuthMiddleware::handle();

        $userId = AuthMiddleware::id();
        $role = AuthMiddleware::role();

        if ($role === 'admin') {
            $tasks = $this->taskModel->findAll();
        } else {
            $tasks = $this->taskModel->findByUser($userId);
        }

        Response::success('Görevler listelendi', $tasks);
    }

    public function show($id) {
        AuthMiddleware::handle();

        $task = $this->taskModel->findById($id);
        
        if (!$task) {
            Response::notFound('Görev bulunamadı');
        }

        // Yetkisiz erişim kontrolü
        $role = AuthMiddleware::role();
        $userId = AuthMiddleware::id();
        
        if ($role !== 'admin' && $task['atanmis_kullanici_id'] != $userId) {
            Response::forbidden('Bu görevi görüntüleme yetkiniz yok');
        }

        Response::success('Görev detayları', $task);
    }

    public function create() {
        AdminMiddleware::handle();

        $request = new Request();
        
        $validation = $request->validate([
            'gorev' => 'required|min:5',
            'atanmis_kullanici_id' => 'required'
        ]);

        if ($validation !== true) {
            Response::error('Doğrulama hatası', $validation);
        }

        $data = [
            'gorev' => $request->input('gorev'),
            'aciklama' => $request->input('aciklama'),
            'son_tarih' => $request->input('son_tarih'),
            'durum' => $request->input('durum', 'bekliyor'),
            'oncelik' => $request->input('oncelik', 'orta'),
            'atayan_admin_id' => AuthMiddleware::id(),
            'atanmis_kullanici_id' => $request->input('atanmis_kullanici_id'),
            'musteri_id' => $request->input('musteri_id')
        ];

        $taskId = $this->taskModel->create($data);
        
        $userId = AuthMiddleware::id();
        $this->logModel->create($userId, 'Görev oluşturma', "Yeni görev ID: $taskId oluşturuldu");

        Response::success('Görev başarıyla oluşturuldu', ['id' => $taskId], 201);
    }

    public function update($id) {
        AuthMiddleware::handle();

        $task = $this->taskModel->findById($id);
        
        if (!$task) {
            Response::notFound('Görev bulunamadı');
        }

        $role = AuthMiddleware::role();
        $userId = AuthMiddleware::id();
        
        // Normal kullanıcı sadece kendi görevinin durumunu değiştirebilir
        if ($role !== 'admin' && $task['atanmis_kullanici_id'] != $userId) {
            Response::forbidden('Bu görevi düzenleme yetkiniz yok');
        }

        $request = new Request();
        
        $data = [];
        
        // Admin tüm alanları güncelleyebilir
        if ($role === 'admin') {
            if ($request->has('gorev')) $data['gorev'] = $request->input('gorev');
            if ($request->has('aciklama')) $data['aciklama'] = $request->input('aciklama');
            if ($request->has('son_tarih')) $data['son_tarih'] = $request->input('son_tarih');
            if ($request->has('oncelik')) $data['oncelik'] = $request->input('oncelik');
            if ($request->has('atanmis_kullanici_id')) $data['atanmis_kullanici_id'] = $request->input('atanmis_kullanici_id');
            if ($request->has('musteri_id')) $data['musteri_id'] = $request->input('musteri_id');
        }
        
        // Durum herkes güncelleyebilir
        if ($request->has('durum')) {
            $data['durum'] = $request->input('durum');
        }

        if (empty($data)) {
            Response::error('Güncellenecek veri bulunamadı');
        }

        $this->taskModel->update($id, $data);
        $this->logModel->create($userId, 'Görev güncelleme', "Görev ID: $id güncellendi");

        Response::success('Görev başarıyla güncellendi');
    }

    public function delete($id) {
        AdminMiddleware::handle();

        $task = $this->taskModel->findById($id);
        
        if (!$task) {
            Response::notFound('Görev bulunamadı');
        }

        $this->taskModel->delete($id);
        
        $userId = AuthMiddleware::id();
        $this->logModel->create($userId, 'Görev silme', "Görev ID: $id silindi");

        Response::success('Görev başarıyla silindi');
    }

    public function stats() {
        AuthMiddleware::handle();

        $userId = AuthMiddleware::id();
        $role = AuthMiddleware::role();
        
        $userIdFilter = $role === 'admin' ? null : $userId;

        $stats = [
            'toplam' => $this->taskModel->countByUser($userIdFilter) ?: $this->taskModel->count(),
            'bekliyor' => $this->taskModel->countByStatus('bekliyor', $userIdFilter),
            'yapiliyor' => $this->taskModel->countByStatus('yapiliyor', $userIdFilter),
            'tamamlandi' => $this->taskModel->countByStatus('tamamlandi', $userIdFilter)
        ];

        Response::success('Görev istatistikleri', $stats);
    }
}