// common.js - Ortak JavaScript Fonksiyonları

const API_URL = '/api';

// Token'ı al
function getToken() {
    return localStorage.getItem('crm_token');
}

// Token'ı kaydet
function setToken(token) {
    localStorage.setItem('crm_token', token);
}

// Token'ı sil
function removeToken() {
    localStorage.removeItem('crm_token');
}

// API isteği yap
async function apiRequest(endpoint, options = {}) {
    const token = getToken();
    
    const config = {
        method: options.method || 'GET',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': token ? `Bearer ${token}` : '',
            ...options.headers
        }
    };
    
    if (options.body) {
        config.body = JSON.stringify(options.body);
    }
    
    try {
        const response = await fetch(`${API_URL}${endpoint}`, config);
        const result = await response.json();
        
        // Token geçersizse login'e yönlendir
        if (response.status === 401) {
            removeToken();
            window.location.href = '/login.html';
            return null;
        }
        
        return result;
    } catch (error) {
        console.error('API Error:', error);
        showToast('Bağlantı hatası!', 'error');
        return null;
    }
}

// Kullanıcı bilgilerini al
async function getCurrentUser() {
    return await apiRequest('/me');
}

// Toast mesaj göster
function showToast(message, type = 'info') {
    // Toast container'ı oluştur (yoksa)
    let container = document.getElementById('toast-container');
    if (!container) {
        container = document.createElement('div');
        container.id = 'toast-container';
        container.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 9999;
        `;
        document.body.appendChild(container);
    }
    
    // Toast elementi oluştur
    const toast = document.createElement('div');
    const bgColor = {
        'success': '#10b981',
        'error': '#ef4444',
        'warning': '#f59e0b',
        'info': '#3b82f6'
    }[type] || '#3b82f6';
    
    toast.style.cssText = `
        background: ${bgColor};
        color: white;
        padding: 15px 20px;
        border-radius: 8px;
        margin-bottom: 10px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        animation: slideIn 0.3s ease;
        min-width: 250px;
        max-width: 400px;
    `;
    
    toast.textContent = message;
    container.appendChild(toast);
    
    // 3 saniye sonra kaldır
    setTimeout(() => {
        toast.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            container.removeChild(toast);
        }, 300);
    }, 3000);
}

// CSS animasyonları ekle
if (!document.getElementById('toast-animations')) {
    const style = document.createElement('style');
    style.id = 'toast-animations';
    style.textContent = `
        @keyframes slideIn {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        
        @keyframes slideOut {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(400px);
                opacity: 0;
            }
        }
    `;
    document.head.appendChild(style);
}

// Tarih formatlama
function formatDate(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    return date.toLocaleDateString('tr-TR', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit'
    });
}

// Tarih ve saat formatlama
function formatDateTime(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    return date.toLocaleString('tr-TR', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Loading göster
function showLoading(elementId) {
    const element = document.getElementById(elementId);
    if (element) {
        element.innerHTML = `
            <div style="text-align: center; padding: 40px;">
                <div style="border: 3px solid #f3f3f3; border-top: 3px solid #667eea; border-radius: 50%; width: 40px; height: 40px; animation: spin 1s linear infinite; margin: 0 auto;"></div>
                <p style="margin-top: 15px; color: #666;">Yükleniyor...</p>
            </div>
        `;
    }
}

// Spin animasyonu
if (!document.getElementById('spin-animation')) {
    const style = document.createElement('style');
    style.id = 'spin-animation';
    style.textContent = `
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    `;
    document.head.appendChild(style);
}

// Çıkış yap
async function logout() {
    if (confirm('Çıkış yapmak istediğinize emin misiniz?')) {
        await apiRequest('/logout', { method: 'POST' });
        removeToken();
        window.location.href = '/login.html';
    }
}

// String'i güvenli HTML'e çevir (XSS koruması)
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Boş değer kontrolü
function isEmpty(value) {
    return value === null || value === undefined || value === '';
}

// Form validasyonu
function validateForm(formId, rules) {
    const form = document.getElementById(formId);
    if (!form) return false;
    
    let isValid = true;
    const errors = [];
    
    for (const [field, rule] of Object.entries(rules)) {
        const input = form.querySelector(`[name="${field}"]`);
        if (!input) continue;
        
        const value = input.value.trim();
        
        // Required kontrolü
        if (rule.required && isEmpty(value)) {
            errors.push(`${rule.label || field} alanı zorunludur`);
            input.style.borderColor = '#ef4444';
            isValid = false;
        } else {
            input.style.borderColor = '#e2e8f0';
        }
        
        // Min length
        if (rule.minLength && value.length < rule.minLength) {
            errors.push(`${rule.label || field} en az ${rule.minLength} karakter olmalıdır`);
            isValid = false;
        }
        
        // Email
        if (rule.email && value && !value.match(/^[^\s@]+@[^\s@]+\.[^\s@]+$/)) {
            errors.push(`${rule.label || field} geçerli bir e-posta adresi olmalıdır`);
            isValid = false;
        }
    }
    
    if (!isValid) {
        showToast(errors[0], 'error');
    }
    
    return isValid;
}

// Onay modalı
function confirm(message, callback) {
    const result = window.confirm(message);
    if (result && callback) {
        callback();
    }
    return result;
}

// Sayfa yüklendiğinde token kontrolü
document.addEventListener('DOMContentLoaded', function() {
    // Login sayfası değilse token kontrolü yap
    if (!window.location.pathname.includes('login.html')) {
        const token = getToken();
        if (!token) {
            window.location.href = '/login.html';
        }
    }
});